#include "Constants.h"
#include "Cpu.h"
#include "Memory.h"
#include "Io.h"
#include "CpuUtilities.h"
#include "CpuEDPrefix.h"


void cpu_instruction__rlc_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint16 val16 = (Uint16)(*reg);
	Uint8 bit7 = val16 >> 7;
	if (bit7) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	val16 = val16 << 1;		// bit 0 := 0
	val16 |= bit7;		// bit 0 := bit 7
	*reg = (Uint8)val16;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__rl_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint8 oldCarry = 0;
	if (cpu_regs()->F & CPU_FLAG_C_CARRY) {
		oldCarry = 1;
	}

	Uint16 temp16 = (Uint16)(*reg);
	Uint8 bit7 = temp16 >> 7;
	if (bit7) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	temp16 = temp16 << 1;		// bit 0 := 0
	temp16 |= oldCarry;
	(*reg) = (Uint8)temp16;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__rrc_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint8 bit0 = *reg & 0x01;
	if (bit0) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	*reg = (*reg) >> 1;
	if (bit0) (*reg) |= 0x80; else (*reg) &= 0x7F;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__rr_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint8 oldCarry = 0;
	if (cpu_regs()->F & CPU_FLAG_C_CARRY) {
		oldCarry = 1;
	}

	Uint16 temp16 = (Uint16)(*reg);
	if (temp16 & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	temp16 = temp16 >> 1;
	if (oldCarry) temp16 |= 0x80; else; temp16 &= 0x7FF;
	(*reg) = (Uint8)temp16;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__sla_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	(*reg) = (*reg) << 1;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__sra_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	if ((*reg) & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	Uint8 bit7 = (*reg) & (1 << 7);
	*reg = (*reg) >> 1;
	*reg |= bit7;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__sll_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	(*reg) = (*reg) << 1;
	(*reg) |= 1;	// put a 1 in bit0

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__srl_reg8(struct instruction* instruction, Uint8* reg, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	if ((*reg) & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	*reg = (*reg) >> 1;

	if ((*reg) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(*reg)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(*reg)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__bit_reg8(struct instruction* instruction, Uint8 bit, Uint8 value8, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	if ((value8 & (1 << bit)) == 0) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;

	cpu_regs()->F |= CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__res_reg8(struct instruction* instruction, Uint8 bit, Uint8* reg8, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint8 shiftedBit = 1 << bit;
	(*reg8) &= ~shiftedBit;

	cpu_mark_instruction_executed(instruction);
}

void cpu_instruction__set_reg8(struct instruction* instruction, Uint8 bit, Uint8* reg8, const char* disassembledName) {
	cpu_finalize_instruction_decode(instruction, 0);
	cpu_set_instruction_disassembled_name_formatted2(instruction, disassembledName, UNUSED, UNUSED);

	if (!cpu_must_execute()) return;
	//execution

	Uint8 shiftedBit = 1 << bit;
	(*reg8) |= shiftedBit;

	cpu_mark_instruction_executed(instruction);
}




void cpu_run_CB_prefix(struct instruction* instruction) {
	Uint8 temp8;

	switch (instruction->opcodeValue) {
	case 0x00:	// rlc b
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->B), "rlc B");
		break;
	case 0x01:	// rlc c
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->C), "rlc C");
		break;
	case 0x02:	// rlc d
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->D), "rlc D");
		break;
	case 0x03:	// rlc e
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->E), "rlc E");
		break;
	case 0x04:	// rlc h
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->H), "rlc H");
		break;
	case 0x05:	// rlc l
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->L), "rlc L");
		break;
	case 0x06:	// rlc (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__rlc_reg8(instruction, &temp8, "rlc (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x07:	// rlc a
		cpu_instruction__rlc_reg8(instruction, &(cpu_regs()->A), "rlc A");
		break;
	case 0x08:	// rrc b
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->B), "rrc B");
		break;
	case 0x09:	// rrc c
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->C), "rrc C");
		break;
	case 0x0A:	// rrc d
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->D), "rrc D");
		break;
	case 0x0B:	// rrc e
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->E), "rrc E");
		break;
	case 0x0C:	// rrc h
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->H), "rrc H");
		break;
	case 0x0D:	// rrc l
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->L), "rrc L");
		break;
	case 0x0E:	// rrc (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__rrc_reg8(instruction, &temp8, "rrc (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x0F:	// rrc a
		cpu_instruction__rrc_reg8(instruction, &(cpu_regs()->A), "rrc A");
		break;
	case 0x10:	// rl b
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->B), "rl B");
		break;
	case 0x11:	// rl c
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->C), "rl C");
		break;
	case 0x12:	// rl d
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->D), "rl D");
		break;
	case 0x13:	// rl e
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->E), "rl E");
		break;
	case 0x14:	// rl h
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->H), "rl H");
		break;
	case 0x15:	// rl l
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->L), "rl L");
		break;
	case 0x16:	// rl (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__rl_reg8(instruction, &temp8, "rl (HL)");

		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x17:	// rl a
		cpu_instruction__rl_reg8(instruction, &(cpu_regs()->A), "rl A");
		break;
	case 0x18:	// rr b
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->B), "rr B");
		break;
	case 0x19:	// rr c
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->C), "rr C");
		break;
	case 0x1A:	// rr d
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->D), "rr D");
		break;
	case 0x1B:	// rr e
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->E), "rr E");
		break;
	case 0x1C:	// rr h
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->H), "rr H");
		break;
	case 0x1D:	// rr l
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->L), "rr L");
		break;
	case 0x1E:	// rr (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__rr_reg8(instruction, &temp8, "rr (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x1F:	// rr a
		cpu_instruction__rr_reg8(instruction, &(cpu_regs()->A), "rr A");
		break;
	case 0x20:	// sla b
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->B), "sla B");
		break;
	case 0x21:	// sla c
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->C), "sla C");
		break;
	case 0x22:	// sla d
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->D), "sla D");
		break;
	case 0x23:	// sla e
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->E), "sla E");
		break;
	case 0x24:	// sla h
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->H), "sla H");
		break;
	case 0x25:	// sla l
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->L), "sla L");
		break;
	case 0x26:	// sla (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__sla_reg8(instruction, &temp8, "sla (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x27:	// sla a
		cpu_instruction__sla_reg8(instruction, &(cpu_regs()->A), "sla A");
		break;
	case 0x28:	// sra b
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->B), "sra B");
		break;
	case 0x29:	// sra c
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->C), "sra C");
		break;
	case 0x2A:	// sra d
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->D), "sra D");
		break;
	case 0x2B:	// sra e
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->E), "sra E");
		break;
	case 0x2C:	// sra h
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->H), "sra H");
		break;
	case 0x2D:	// sra l
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->L), "sra L");
		break;
	case 0x2E:	// sra (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__sra_reg8(instruction, &temp8, "sra (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x2F:	// sra a
		cpu_instruction__sra_reg8(instruction, &(cpu_regs()->A), "sra A");
		break;
	case 0x30:	// sll b
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->B), "sll B");
		break;
	case 0x31:	// sll c
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->C), "sll C");
		break;
	case 0x32:	// sll d
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->D), "sll D");
		break;
	case 0x33:	// sll e
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->E), "sll E");
		break;
	case 0x34:	// sll h
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->H), "sll H");
		break;
	case 0x35:	// sll l
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->L), "sll L");
		break;
	case 0x36:	// sll (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__sll_reg8(instruction, &temp8, "sll (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x37:	// sll a
		cpu_instruction__sll_reg8(instruction, &(cpu_regs()->A), "sll A");
		break;
	case 0x38:	// srl b
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->B), "srl B");
		break;
	case 0x39:	// srl c
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->C), "srl C");
		break;
	case 0x3A:	// srl d
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->D), "srl D");
		break;
	case 0x3B:	// srl e
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->E), "srl E");
		break;
	case 0x3C:	// srl h
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->H), "srl H");
		break;
	case 0x3D:	// srl l
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->L), "srl L");
		break;
	case 0x3E:	// srl (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__srl_reg8(instruction, &temp8, "srl (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x3F:	// srl a
		cpu_instruction__srl_reg8(instruction, &(cpu_regs()->A), "srl A");
		break;
	case 0x40:	// bit 0, b
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->B, "bit 0, B");
		break;
	case 0x41:	// bit 0, c
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->C, "bit 0, C");
		break;
	case 0x42:	// bit 0, d
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->D, "bit 0, D");
		break;
	case 0x43:	// bit 0, e
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->E, "bit 0, E");
		break;
	case 0x44:	// bit 0, h
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->H, "bit 0, H");
		break;
	case 0x45:	// bit 0, l
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->L, "bit 0, L");
		break;
	case 0x46:	// bit 0, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 0, temp8, "bit 0, (HL)");
		break;
	case 0x47:	// bit 0, a
		cpu_instruction__bit_reg8(instruction, 0, cpu_regs()->A, "bit 0, A");
		break;
	case 0x48:	// bit 1, b
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->B, "bit 1, B");
		break;
	case 0x49:	// bit 1, c
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->C, "bit 1, C");
		break;
	case 0x4A:	// bit 1, d
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->D, "bit 1, D");
		break;
	case 0x4B:	// bit 1, e
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->E, "bit 1, E");
		break;
	case 0x4C:	// bit 1, h
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->H, "bit 1, H");
		break;
	case 0x4D:	// bit 1, l
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->L, "bit 1, L");
		break;
	case 0x4E:	// bit 1, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 1, temp8, "bit 1, (HL)");
		break;
	case 0x4F:	// bit 1, a
		cpu_instruction__bit_reg8(instruction, 1, cpu_regs()->A, "bit 1, A");
		break;
	case 0x50:	// bit 2, b
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->B, "bit 2, B");
		break;
	case 0x51:	// bit 2, c
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->C, "bit 2, C");
		break;
	case 0x52:	// bit 2, d
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->D, "bit 2, D");
		break;
	case 0x53:	// bit 2, e
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->E, "bit 2, E");
		break;
	case 0x54:	// bit 2, h
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->H, "bit 2, H");
		break;
	case 0x55:	// bit 2, l
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->L, "bit 2, L");
		break;
	case 0x56:	// bit 2, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 2, temp8, "bit 2, (HL)");
		break;
	case 0x57:	// bit 2, a
		cpu_instruction__bit_reg8(instruction, 2, cpu_regs()->A, "bit 2, A");
		break;
	case 0x58:	// bit 3, b
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->B, "bit 3, B");
		break;
	case 0x59:	// bit 3, c
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->C, "bit 3, C");
		break;
	case 0x5A:	// bit 3, d
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->D, "bit 3, D");
		break;
	case 0x5B:	// bit 3, e
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->E, "bit 3, E");
		break;
	case 0x5C:	// bit 3, h
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->H, "bit 3, H");
		break;
	case 0x5D:	// bit 3, l
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->L, "bit 3, L");
		break;
	case 0x5E:	// bit 3, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 3, temp8, "bit 3, (HL)");
		break;
	case 0x5F:	// bit 3, a
		cpu_instruction__bit_reg8(instruction, 3, cpu_regs()->A, "bit 3, A");
		break;
	case 0x60:	// bit 4, b
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->B, "bit 4, B");
		break;
	case 0x61:	// bit 4, c
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->C, "bit 4, C");
		break;
	case 0x62:	// bit 4, d
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->D, "bit 4, D");
		break;
	case 0x63:	// bit 4, e
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->E, "bit 4, E");
		break;
	case 0x64:	// bit 4, h
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->H, "bit 4, H");
		break;
	case 0x65:	// bit 4, l
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->L, "bit 4, L");
		break;
	case 0x66:	// bit 4, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 4, temp8, "bit 4, (HL)");
		break;
	case 0x67:	// bit 4, a
		cpu_instruction__bit_reg8(instruction, 4, cpu_regs()->A, "bit 4, A");
		break;
	case 0x68:	// bit 5, b
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->B, "bit 5, B");
		break;
	case 0x69:	// bit 5, c
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->C, "bit 5, C");
		break;
	case 0x6A:	// bit 5, d
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->D, "bit 5, D");
		break;
	case 0x6B:	// bit 5, e
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->E, "bit 5, E");
		break;
	case 0x6C:	// bit 5, h
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->H, "bit 5, H");
		break;
	case 0x6D:	// bit 5, l
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->L, "bit 5, L");
		break;
	case 0x6E:	// bit 5, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 5, temp8, "bit 5, (HL)");
		break;
	case 0x6F:	// bit 5, a
		cpu_instruction__bit_reg8(instruction, 5, cpu_regs()->A, "bit 5, A");
		break;
	case 0x70:	// bit 6, b
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->B, "bit 6, B");
		break;
	case 0x71:	// bit 6, c
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->C, "bit 6, C");
		break;
	case 0x72:	// bit 6, d
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->D, "bit 6, D");
		break;
	case 0x73:	// bit 6, e
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->E, "bit 6, E");
		break;
	case 0x74:	// bit 6, h
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->H, "bit 6, H");
		break;
	case 0x75:	// bit 6, l
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->L, "bit 6, L");
		break;
	case 0x76:	// bit 6, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 6, temp8, "bit 6, (HL)");
		break;
	case 0x77:	// bit 6, a
		cpu_instruction__bit_reg8(instruction, 6, cpu_regs()->A, "bit 6, A");
		break;
	case 0x78:	// bit 7, b
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->B, "bit 7, B");
		break;
	case 0x79:	// bit 7, c
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->C, "bit 7, C");
		break;
	case 0x7A:	// bit 7, d
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->D, "bit 7, D");
		break;
	case 0x7B:	// bit 7, e
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->E, "bit 7, E");
		break;
	case 0x7C:	// bit 7, h
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->H, "bit 7, H");
		break;
	case 0x7D:	// bit 7, l
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->L, "bit 7, L");
		break;
	case 0x7E:	// bit 7, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 4;
		cpu_instruction__bit_reg8(instruction, 7, temp8, "bit 7, (HL)");
		break;
	case 0x7F:	// bit 7, a
		cpu_instruction__bit_reg8(instruction, 7, cpu_regs()->A, "bit 7, A");
		break;
	case 0x80:	// res 0, b
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->B), "res 0, B");
		break;
	case 0x81:	// res 0, c
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->C), "res 0, C");
		break;
	case 0x82:	// res 0, d
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->D), "res 0, D");
		break;
	case 0x83:	// res 0, e
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->E), "res 0, E");
		break;
	case 0x84:	// res 0, h
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->H), "res 0, H");
		break;
	case 0x85:	// res 0, l
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->L), "res 0, L");
		break;
	case 0x86:	// res 0, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 0, &temp8, "res 0, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x87:	// res 0, a
		cpu_instruction__res_reg8(instruction, 0, &(cpu_regs()->A), "res 0, A");
		break;
	case 0x88:	// res 1, b
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->B), "res 1, B");
		break;
	case 0x89:	// res 1, c
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->C), "res 1, C");
		break;
	case 0x8A:	// res 1, d
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->D), "res 1, D");
		break;
	case 0x8B:	// res 1, e
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->E), "res 1, E");
		break;
	case 0x8C:	// res 1, h
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->H), "res 1, H");
		break;
	case 0x8D:	// res 1, l
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->L), "res 1, L");
		break;
	case 0x8E:	// res 1, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 1, &temp8, "res 1, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x8F:	// res 1, a
		cpu_instruction__res_reg8(instruction, 1, &(cpu_regs()->A), "res 1, A");
		break;
	case 0x90:	// res 2, b
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->B), "res 2, B");
		break;
	case 0x91:	// res 2, c
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->C), "res 2, C");
		break;
	case 0x92:	// res 2, d
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->D), "res 2, D");
		break;
	case 0x93:	// res 2, e
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->E), "res 2, E");
		break;
	case 0x94:	// res 2, h
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->H), "res 2, H");
		break;
	case 0x95:	// res 2, l
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->L), "res 2, L");
		break;
	case 0x96:	// res 2, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 2, &temp8, "res 2, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x97:	// res 2, a
		cpu_instruction__res_reg8(instruction, 2, &(cpu_regs()->A), "res 2, A");
		break;
	case 0x98:	// res 3, b
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->B), "res 3, B");
		break;
	case 0x99:	// res 3, c
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->C), "res 3, C");
		break;
	case 0x9A:	// res 3, d
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->D), "res 3, D");
		break;
	case 0x9B:	// res 3, e
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->E), "res 3, E");
		break;
	case 0x9C:	// res 3, h
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->H), "res 3, H");
		break;
	case 0x9D:	// res 3, l
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->L), "res 3, L");
		break;
	case 0x9E:	// res 3, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 3, &temp8, "res 3, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0x9F:	// res 3, a
		cpu_instruction__res_reg8(instruction, 3, &(cpu_regs()->A), "res 3, A");
		break;
	case 0xA0:	// res 4, b
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->B), "res 4, B");
		break;
	case 0xA1:	// res 4, c
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->C), "res 4, C");
		break;
	case 0xA2:	// res 4, d
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->D), "res 4, D");
		break;
	case 0xA3:	// res 4, e
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->E), "res 4, E");
		break;
	case 0xA4:	// res 4, h
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->H), "res 4, H");
		break;
	case 0xA5:	// res 4, l
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->L), "res 4, L");
		break;
	case 0xA6:	// res 4, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 4, &temp8, "res 4, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xA7:	// res 4, a
		cpu_instruction__res_reg8(instruction, 4, &(cpu_regs()->A), "res 4, A");
		break;
	case 0xA8:	// res 5, b
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->B), "res 5, B");
		break;
	case 0xA9:	// res 5, c
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->C), "res 5, C");
		break;
	case 0xAA:	// res 5, d
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->D), "res 5, D");
		break;
	case 0xAB:	// res 5, e
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->E), "res 5, E");
		break;
	case 0xAC:	// res 5, h
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->H), "res 5, H");
		break;
	case 0xAD:	// res 5, l
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->L), "res 5, L");
		break;
	case 0xAE:	// res 5, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 5, &temp8, "res 5, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xAF:	// res 5, a
		cpu_instruction__res_reg8(instruction, 5, &(cpu_regs()->A), "res 5, A");
		break;
	case 0xB0:	// res 6, b
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->B), "res 6, B");
		break;
	case 0xB1:	// res 6, c
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->C), "res 6, C");
		break;
	case 0xB2:	// res 6, d
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->D), "res 6, D");
		break;
	case 0xB3:	// res 6, e
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->E), "res 6, E");
		break;
	case 0xB4:	// res 6, h
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->H), "res 6, H");
		break;
	case 0xB5:	// res 6, l
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->L), "res 6, L");
		break;
	case 0xB6:	// res 6, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 6, &temp8, "res 6, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xB7:	// res 6, a
		cpu_instruction__res_reg8(instruction, 6, &(cpu_regs()->A), "res 6, A");
		break;
	case 0xB8:	// res 7, b
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->B), "res 7, B");
		break;
	case 0xB9:	// res 7, c
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->C), "res 7, C");
		break;
	case 0xBA:	// res 7, d
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->D), "res 7, D");
		break;
	case 0xBB:	// res 7, e
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->E), "res 7, E");
		break;
	case 0xBC:	// res 7, h
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->H), "res 7, H");
		break;
	case 0xBD:	// res 7, l
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->L), "res 7, L");
		break;
	case 0xBE:	// res 7, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__res_reg8(instruction, 7, &temp8, "res 7, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xBF:	// res 7, a
		cpu_instruction__res_reg8(instruction, 7, &(cpu_regs()->A), "res 7, A");
		break;
	case 0xC0:	// set 0, b
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->B), "set 0, B");
		break;
	case 0xC1:	// set 0, c
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->C), "set 0, C");
		break;
	case 0xC2:	// set 0, d
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->D), "set 0, D");
		break;
	case 0xC3:	// set 0, e
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->E), "set 0, E");
		break;
	case 0xC4:	// set 0, h
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->H), "set 0, H");
		break;
	case 0xC5:	// set 0, l
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->L), "set 0, L");
		break;
	case 0xC6:	// set 0, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 0, &temp8, "set 0, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xC7:	// set 0, a
		cpu_instruction__set_reg8(instruction, 0, &(cpu_regs()->A), "set 0, A");
		break;
	case 0xC8:	// set 1, b
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->B), "set 1, B");
		break;
	case 0xC9:	// set 1, c
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->C), "set 1, C");
		break;
	case 0xCA:	// set 1, d
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->D), "set 1, D");
		break;
	case 0xCB:	// set 1, e
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->E), "set 1, E");
		break;
	case 0xCC:	// set 1, h
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->H), "set 1, H");
		break;
	case 0xCD:	// set 1, l
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->L), "set 1, L");
		break;
	case 0xCE:	// set 1, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 1, &temp8, "set 1, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xCF:	// set 1, a
		cpu_instruction__set_reg8(instruction, 1, &(cpu_regs()->A), "set 1, A");
		break;
	case 0xD0:	// set 2, b
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->B), "set 2, B");
		break;
	case 0xD1:	// set 2, c
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->C), "set 2, C");
		break;
	case 0xD2:	// set 2, d
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->D), "set 2, D");
		break;
	case 0xD3:	// set 2, e
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->E), "set 2, E");
		break;
	case 0xD4:	// set 2, h
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->H), "set 2, H");
		break;
	case 0xD5:	// set 2, l
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->L), "set 2, L");
		break;
	case 0xD6:	// set 2, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 2, &temp8, "set 2, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xD7:	// set 2, a
		cpu_instruction__set_reg8(instruction, 2, &(cpu_regs()->A), "set 2, A");
		break;
	case 0xD8:	// set 3, b
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->B), "set 3, B");
		break;
	case 0xD9:	// set 3, c
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->C), "set 3, C");
		break;
	case 0xDA:	// set 3, d
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->D), "set 3, D");
		break;
	case 0xDB:	// set 3, e
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->E), "set 3, E");
		break;
	case 0xDC:	// set 3, h
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->H), "set 3, H");
		break;
	case 0xDD:	// set 3, l
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->L), "set 3, L");
		break;
	case 0xDE:	// set 3, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 3, &temp8, "set 3, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xDF:	// set 3, a
		cpu_instruction__set_reg8(instruction, 3, &(cpu_regs()->A), "set 3, A");
		break;
	case 0xE0:	// set 4, b
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->B), "set 4, B");
		break;
	case 0xE1:	// set 4, c
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->C), "set 4, C");
		break;
	case 0xE2:	// set 4, d
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->D), "set 4, D");
		break;
	case 0xE3:	// set 4, e
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->E), "set 4, E");
		break;
	case 0xE4:	// set 4, h
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->H), "set 4, H");
		break;
	case 0xE5:	// set 4, l
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->L), "set 4, L");
		break;
	case 0xE6:	// set 4, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 4, &temp8, "set 4, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xE7:	// set 4, a
		cpu_instruction__set_reg8(instruction, 4, &(cpu_regs()->A), "set 4, A");
		break;
	case 0xE8:	// set 5, b
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->B), "set 5, B");
		break;
	case 0xE9:	// set 5, c
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->C), "set 5, C");
		break;
	case 0xEA:	// set 5, d
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->D), "set 5, D");
		break;
	case 0xEB:	// set 5, e
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->E), "set 5, E");
		break;
	case 0xEC:	// set 5, h
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->H), "set 5, H");
		break;
	case 0xED:	// set 5, l
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->L), "set 5, L");
		break;
	case 0xEE:	// set 5, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 5, &temp8, "set 5, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xEF:	// set 5, a
		cpu_instruction__set_reg8(instruction, 5, &(cpu_regs()->A), "set 5, A");
		break;
	case 0xF0:	// set 6, b
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->B), "set 6, B");
		break;
	case 0xF1:	// set 6, c
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->C), "set 6, C");
		break;
	case 0xF2:	// set 6, d
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->D), "set 6, D");
		break;
	case 0xF3:	// set 6, e
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->E), "set 6, E");
		break;
	case 0xF4:	// set 6, h
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->H), "set 6, H");
		break;
	case 0xF5:	// set 6, l
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->L), "set 6, L");
		break;
	case 0xF6:	// set 6, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 6, &temp8, "set 6, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xF7:	// set 6, a
		cpu_instruction__set_reg8(instruction, 6, &(cpu_regs()->A), "set 6, A");
		break;
	case 0xF8:	// set 7, b
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->B), "set 7, B");
		break;
	case 0xF9:	// set 7, c
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->C), "set 7, C");
		break;
	case 0xFA:	// set 7, d
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->D), "set 7, D");
		break;
	case 0xFB:	// set 7, e
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->E), "set 7, E");
		break;
	case 0xFC:	// set 7, h
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->H), "set 7, H");
		break;
	case 0xFD:	// set 7, l
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->L), "set 7, L");
		break;
	case 0xFE:	// set 7, (hl)
		temp8 = memory_read8(*cpu_regs()->HL);
		instruction->tstatesElapsed += 7;
		cpu_instruction__set_reg8(instruction, 7, &temp8, "set 7, (HL)");
		if (!cpu_must_execute()) break;

		memory_write8(*cpu_regs()->HL, temp8);
		break;
	case 0xFF:	// set 7, a
		cpu_instruction__set_reg8(instruction, 7, &(cpu_regs()->A), "set 7, A");
		break;

	default:
		cpu_finalize_instruction_decode(instruction, 0);
		cpu_mark_instruction_unknown(instruction);
		break;
	}
}
