#include <stdlib.h>

#include "Constants.h"
#include "DataBus.h"
#include "Memory.h"

static Uint8* _memory = NULL;
static func_on_memory_write_callback* _onMemoryWriteCallback;

void on_memory_write_NOOP_callback(Uint16 address) {
}

void memory_start(func_on_memory_write_callback onMemoryWriteCallback) {
	if (_memory != NULL) {
		memory_destroy();
	}
	_memory = (Uint8*)malloc(SPECTRUM_ROM_SIZE + SPECTRUM_RAM_SIZE);
	if (_memory == NULL) {
		return;
	}

	_onMemoryWriteCallback = on_memory_write_NOOP_callback;

	if (onMemoryWriteCallback != NULL) {
		_onMemoryWriteCallback = onMemoryWriteCallback;
	}

	// initialize memory bytes
	for (int i = 0; i < SPECTRUM_ROM_SIZE + SPECTRUM_RAM_SIZE; i++) {
		_memory[i] = 0x00;
	}
}

void memory_destroy() {
	if (_memory != NULL) {
		free(_memory);
		_memory = NULL;
	}

	_onMemoryWriteCallback = NULL;
}

void memory_load(Uint8* source, Uint16 offset, Uint16 size) {
	if (offset + size > SPECTRUM_ROM_SIZE + SPECTRUM_RAM_SIZE) {
		return;
	}
	SDL_memcpy(_memory + offset, source, size);
}

Uint8 memory_read8(Uint16 address) {
	Uint8 value = _memory[address & 0xffff];
	return value;
}

Uint16 memory_read16(Uint16 address) {
	Uint8 msb = _memory[(address + 1) & 0xffff];
	data_bus_write8(msb);

	return _memory[address & 0xffff] + (msb << 8);
}

void memory_write8(Uint16 address, Uint8 value) {
	data_bus_write8(value);

	if (address < SPECTRUM_ROM_SIZE) {
		return;
	}

	_memory[address] = value;

	_onMemoryWriteCallback(address);
}

void memory_write16(Uint16 address, Uint16 value) {
	memory_write8(address, (Uint8)value);			// LSB
	memory_write8(address+1, (Uint8)(value >> 8));	// MSB
}

void memory_dump(Uint8* destination, Uint16 startAddress, Uint16 size)
{
	while (size--) {
		*destination++ = _memory[startAddress++];
	}
}